//  Copyright (C) 2002-2003 Intel Corporation, All Rights Reserved.
//  Permission is hereby granted to merge this program code with 
//  other program material to create a derivative work.  This 
//  derivative work may be distributed in compiled object form only.
//  Any other publication of this program, in any form, without the 
//  explicit permission of the copyright holder is prohibited.
//
//  Send questions and comments to erik.j.johnson@intel.com, 
//  aaron.kunze@intel.com

//-------------------------------------------------------------------
// tx.uc - Chapter 5
// Segments packets and transmits
// them using a single thread onto a single port
//

#ifndef spi3_TX_UC
#define spi3_TX_UC

#include "tx.h"
#include "stdmac.uc"

#define TBUF_ELEM_SIZE			64
#define TBUF_ELEM_SIZE_SHIFT	6
//#define NUM_TBUFS				((128 * 64 / TBUF_ELEM_SIZE) / 4)
#define NUM_TBUFS				(16)

#define TX_CONTROL_VAL			((1 << TX_EN_SPHY_BITPOS) | \
								 (1 << TX_ENABLE_BITPOS)	| \
								 (0 << TBUF_ELE_SIZE_0_BITPOS))


//-------------------------------------------------------------------
// This state is used to store the current packet's segmentation
// state
.reg tx_sop tx_eop // 1 if the current mpacket is SOP/EOP respectively
.reg tx_cur_mpacket_addr // A pointer to the current mpacket
.reg tx_remaining_length // Length remaining, in bytes, of the 
						 // current mpacket
.reg tx_cur_buf_handle // The handle of the current buffer


//-------------------------------------------------------------------
// spi3_tx_init
//
//    Description:
//       Initialize the appropriate CSRs to transmit packets on
//       the spi3 interface.
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n/a
//      Inputs: n/a
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: Writes to the MSF transmit control CSR
//
//    See also: n/a
//
#macro spi3_tx_init()
.begin
	.if (ctx() == 0)
		.reg addr // Holds the address of the CSR being set

		.reg $tx_ctl_xfer
		.sig msf_tx_ctl_sig

		//----------- Set the transmit control CSR in the MSF.
		immed32($tx_ctl_xfer, 0x0f080004)
		immed32(addr, MSF_TX_CONTROL_ADDR)
		msf[write, $tx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_tx_ctl_sig]
		immed32($tx_ctl_xfer, 0xff080004)
		immed32(addr, MSF_TX_CONTROL_ADDR)
		msf[write, $tx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_tx_ctl_sig]


		//----------- Set the tx up CSRs in the MSF.
		immed32($tx_ctl_xfer, 0x19)
		immed32(addr, 0x90)
		msf[write, $tx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_tx_ctl_sig]
		immed32($tx_ctl_xfer, 0x19)
		immed32(addr, 0x94)
		msf[write, $tx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_tx_ctl_sig]
		immed32($tx_ctl_xfer, 0x19)
		immed32(addr, 0x98)
		msf[write, $tx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_tx_ctl_sig]
		immed32($tx_ctl_xfer, 0x19)
		immed32(addr, 0x9c)
		msf[write, $tx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_tx_ctl_sig]

	.endif
.end
#endm


//-------------------------------------------------------------------
// _spi3_tx_move_dram_to_tbuf
//
//    Description:
//       Transfer the given DRAM memory into the given TBUF element
//
//    Parameters:
//      Outputs: n/a 
//      In/Outs: n/a
//      Inputs: in_tbuf_elem - The TBUF element number to use
//              in_dram_addr - The start address to the packet to
//							   transfer
//				in_size - The number of bytes to transfer
//				in_dram_sig	 - The signal to use for the transfer
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: The given dual signal must be caught by the
//					calling routine.
//					The in_size must be between 1 and
//					128.  The number of bytes transfered will
//					be (in_size >> 3) << 3) since
//					the transfer must be an even number of quadwords
//    See also: n/a
//
//    Example Usage:
//		.sig dram_signal
//		tx_move_dram_to_tbuf(2, 0x20, 64, dram_signal)
//
#macro _spi3_tx_move_dram_to_tbuf(\
	in_tbuf_elem, \
	in_dram_addr, \
	in_size, \
	in_dram_sig)
.begin
	.reg indir tbuf_addr qwords_to_xfer new_size

	// Compute the TBUF address.  This is the base TBUF
	// address in the MSF plus the element number times
	// 64.
	immed32(tbuf_addr, MSF_TBUF_BASE_ADDR)
	alu_shf_left(tbuf_addr, tbuf_addr, OR,
		 		 in_tbuf_elem, 6)
	// Override the tbuf address
	alu[indir, --, B, 1, <<4] 
	alu[indir, indir, OR, tbuf_addr, <<5]
	// Override the transfer size
	alu_shf_left(indir, indir, OR, 1, 25)
	add(new_size, in_size, 7)
	alu_shf_right(qwords_to_xfer, 0xff, AND, new_size, 3)
	sub(qwords_to_xfer, qwords_to_xfer, 1) 
	alu_shf_left(indir, indir, OR, qwords_to_xfer, 21)
	dram[tbuf_wr, --, in_dram_addr, 0, 8], 
		indirect_ref, 
		sig_done[in_dram_sig]
	.use indir // Suppress an assembler warning
.end
#endm


//-------------------------------------------------------------------
// _spi3_tx_validate_tbuf
//
//    Description:
//       Validate the given TBUF entry
//
//    Parameters:
//      Outputs: n/a 
//      In/Outs: n/a
//      Inputs: in_tbuf_elem - The TBUF element number to use
//              in_sop       - 1 if the TBUF is an SOP, 0 otherwise
//              in_eop       - 1 if the TBUF is an EOP, 0 otherwise
//				in_size - The number of bytes to transfer
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: The given signal must be caught by the
//					calling routine.
//    See also: n/a
//
//
#macro _spi3_tx_validate_tbuf(\
	in_tbuf_elem, \
	in_sop, \
	in_eop, \
	in_size)
.begin
	.reg $tbuf_control_xfer $reserved tbuf_control
	.xfer_order $tbuf_control_xfer $reserved 
	.reg tbuf_addr
	.sig msf_sig

	// Set the mpacket length
	shf_left(tbuf_control, in_size, 24)
	// Set SOP and EOP
	alu_shf_left(tbuf_control, tbuf_control, OR, 
				 in_sop, 9)
	alu_shf_left(tbuf_control, tbuf_control, OR, 
				 in_eop, 8)

	immed32(tbuf_addr, MSF_TBUF_CONTROL_BASE_ADDR)
	alu_shf_left(tbuf_addr, tbuf_addr, OR,
				 in_tbuf_elem, 3)

	move($tbuf_control_xfer, tbuf_control)
	immed32($reserved, 0)
	msf[write, $tbuf_control_xfer, tbuf_addr, 0, 2], 
		ctx_swap[msf_sig]
.end
#endm

//-------------------------------------------------------------------
// _spi3_tx_update_tbufs_in_flight
//
//    Description:
//       Read the current TX sequence number and with the last
//		 sequence number value, update the number of tbuf
//		 elements in flight.
//
//    Parameters:
//      Outputs: n/a 
//      In/Outs: io_tbuf_in_flight - On input, the last number of
//                                   TBUFs in flight based on
//                                   the last read value of
//                                   io_last_tx_seq.  On output,
//                                   the number of tbufs in flight
//                                   according to the updated
//                                   io_last_tx_seq
//               io_last_tx_seq    - On input, the last value
//                                   read for the TX sequence number.
//                                   On output, the current value
//                                   of the TX sequence number
//      Inputs: n/a
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: n/a
//    See also: n/a
//
//
#macro _spi3_tx_update_tbufs_in_flight(\
	io_tbufs_in_flight, \
	io_last_tx_seq)
.begin
	.reg $cur_tx_seq_xfer cur_tx_seq addr tbufs_used
	.sig msf_sig

	// First read the current sequence number
	immed32(addr, MSF_TX_SEQUENCE_0_ADDR)
	msf[read, $cur_tx_seq_xfer, addr, 0, 1], 
		ctx_swap[msf_sig]
	alu_op(cur_tx_seq, 0xff, AND, $cur_tx_seq_xfer)

	// Compute how many TBUFs have been consumed
	// since the last read.  Account for wrap
	.if (io_last_tx_seq <= cur_tx_seq)
		sub(tbufs_used, cur_tx_seq, io_last_tx_seq)
	.else
		sub(tbufs_used, io_last_tx_seq, cur_tx_seq)
	.endif

	// Subtract the tbufs_used from the current 
	// number of tbufs in flight
	sub(io_tbufs_in_flight, io_tbufs_in_flight, 
		tbufs_used)

	// Save the sequence number
	move(io_last_tx_seq, cur_tx_seq)
.end
#endm


//-------------------------------------------------------------------
// _spi3_tx_get_and_update_state
//
//    Description:
//       Get and update the segmentation state associated with a
//		 particular TBUF element.  This version of the routine first
//       attempts to extract the state from global registers, if that
//       state has the EOP state set, then a new packet is dequeued
//       and the first mpacket for the new packet is returned. 
//
//    Parameters:
//      Outputs: out_buf_handle - The handle of the current buffer
//				 out_mpkt_addr - A pointer to the current mpacket
//                               to transmit
//               out_mpkt_length - The length (in bytes) of the
//                                 mpacket
//               out_sop - 1 if the mpacket is the start-of-pcaket,
//                         0 otherwise
//               out_eop - 1 if the mpacket is the end-of-pcaket,
//                         0 otherwise
//      In/Outs: n/a
//      Inputs: in_next_tbuf_elem - The TBUF element for which 
//                                  to retrieve the state
//      Constants: n/a
//      Labels: n/a
//
//    Side effects:  
//
//      This routine does not return until valid state can be obtained
//      This routine currently ignored the given TBUF element
//
//    See also: n/a
//
//
#macro _spi3_tx_get_and_update_state(\
	out_buf_handle, \
	out_mpkt_addr, \
	out_mpkt_length, \
	out_sop, \
	out_eop, \
	in_next_tbuf_elem)
.begin
	// If EOP is true, get a new packet
	.if (tx_eop)
		.while (1)
		.begin
			// Dequeue a packet from the processing task
			dl_source();
						
			// Check for an empty queue
			.if (dl_buf_handle != 0)
				immed32(tx_sop, 1)
				move(tx_cur_buf_handle, dl_buf_handle)
				dl_buf_get_data(tx_cur_mpacket_addr, 
								dl_buf_handle)
				dl_meta_get_buffer_size(
								tx_remaining_length)
				.break
			.endif
		.end
		.endw
	.endif

	move(out_mpkt_addr, tx_cur_mpacket_addr)
	move(out_buf_handle, tx_cur_buf_handle)
	move(out_sop, tx_sop)

	// Update the global state for the next call to 
	// this macro.  Check for EOP
	.if (tx_remaining_length <= TBUF_ELEM_SIZE)
		immed32(tx_eop, 1)
		move(out_mpkt_length, tx_remaining_length)
	.else
		immed32(tx_eop, 0)
		move(out_mpkt_length, TBUF_ELEM_SIZE)
	.endif

	add(tx_cur_mpacket_addr, tx_cur_mpacket_addr, 
		TBUF_ELEM_SIZE)
	sub(tx_remaining_length, tx_remaining_length, 
		TBUF_ELEM_SIZE)
			
	immed32(tx_sop, 0)
	move(out_eop, tx_eop)
.end
#endm


//-------------------------------------------------------------------
// spi3_tx
//
//    Description:
//		 Segment and transmit buffers.  This microblock operates
//		 as a context-pipeline and thus never returns
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n/a
//      Inputs: n/a
//      Constants: n/a
//      Labels: n/a
//
//    Side effects:  
//
//    See also: n/a
//
//
#macro spi3_tx()
.begin
	// State used during the transmission to ensure
	// the TBUFs are used in order and without 
	// overrunning the hardware
	.reg next_tbuf_elem   // The index of the next tbuf 
	 			   		  // element to be used
	.reg last_tx_seq	  // The value of the last read 
						  // to the tx_sequence number
	.reg tbufs_in_flight  // The number of TBUFs 
						  // currently being transmitted


	// State associated with the current mpacket
	.reg sop eop
	.reg mpkt_addr mpkt_length
	.sig dram_to_tbuf_sig

	// Initialize the transmit state
	immed32(next_tbuf_elem, 0)
	immed32(tbufs_in_flight, 0)
	immed32(last_tx_seq, 0)

	// Setting the global EOP = 1 will force a 
	// dequeue
	immed32(tx_eop, 1)

	// Suppress assembler warnings
	.set tx_cur_mpacket_addr tx_remaining_length 
	.set tx_cur_buf_handle tx_sop

	.while(1)
		// Check that the TBUF is available for use
		.while (tbufs_in_flight == NUM_TBUFS)
			// We are out of TBUFs, wait for the
			// sequence number to increase
			_spi3_tx_update_tbufs_in_flight(
				tbufs_in_flight,
				last_tx_seq)
		.endw

		// Get the state (next mpacket) for the 
		// current TBUF element.
		_spi3_tx_get_and_update_state(
				dl_buf_handle, 
				mpkt_addr, 
				mpkt_length, 
				sop, eop, 
				next_tbuf_elem)

		// Move the next portion of the packet into 
		// the next tbuf
		_spi3_tx_move_dram_to_tbuf(
			next_tbuf_elem, 
			mpkt_addr,
			mpkt_length, 
			dram_to_tbuf_sig)

		// As an optimization, if we have only one 
		// more TBUF available (after this one), then 
		// read and update the tbufs in fight during 
		// the transfer from DRAM to TBUF
		.if (tbufs_in_flight == (NUM_TBUFS - 1))
			_spi3_tx_update_tbufs_in_flight(
				tbufs_in_flight,
			 	last_tx_seq)
		.endif

		// Wait for the TBUF to be filled
		ctx_arb[dram_to_tbuf_sig]

		// Write the TBUF control word to validate 
		// the entry
		_spi3_tx_validate_tbuf(
			next_tbuf_elem, 
			sop, 
			eop, 
			mpkt_length)

		// Update the global transmit state
		add(next_tbuf_elem, next_tbuf_elem, 
			(TBUF_ELEM_SIZE / 64))
		.if (next_tbuf_elem >= NUM_TBUFS)
			immed32(next_tbuf_elem, 0)
		.endif

		add(tbufs_in_flight, tbufs_in_flight, 1)

		.if (eop)
			// Free the buffer
			dl_buf_drop(dl_buf_handle)
		.endif
	.endw
.end
#endm

#endif // spi3_TX_UC