//  Copyright (C) 2002-2003 Intel Corporation, All Rights Reserved.
//  Permission is hereby granted to merge this program code with 
//  other program material to create a derivative work.  This 
//  derivative work may be distributed in compiled object form only.
//  Any other publication of this program, in any form, without the 
//  explicit permission of the copyright holder is prohibited.
//
//  Send questions and comments to erik.j.johnson@intel.com, 
//  aaron.kunze@intel.com

//-------------------------------------------------------------------
// scratch_rings.h - Chapter 5
// Common values for the packet rings used between rx, processing
// and tranxmit
//

#ifndef SCRATCH_RINGS_H
#define SCRATCH_RINGS_H


#define RING_BASE_SIZE_BITPOS			30

#define RING_DATA_SIZE_BYTES			16

//-------------------------------------------------------------------
// MicroC data structures
#ifdef MICRO_C

#include "dl_buf.h"

// This structure represents the data placed on the rx->processing
// and processing->tx rings.
typedef struct s_ring_data
{
	dl_buf_handle_t		handle;
	unsigned int		length;
	unsigned int		offset;
	unsigned int		sequence;
} ring_data_t;


// Function prototypes

//-------------------------------------------------------------------
// scratch_ring_init
//
//    Description:
//       Initialize the given scratch ring
void scratch_ring_init(
	unsigned int ring_number,
	unsigned int ring_base, 
	unsigned int ring_size);


//-------------------------------------------------------------------
// scratch_ring_empty
//
//    Description:
//       Check if the given scratchpad ring is empty.
bool scratch_ring_empty(unsigned int ring_number);


//-------------------------------------------------------------------
// scratch_ring_get_buffer
//
//    Description:
//       Dequeue a buffer, length and offset tuple from the given
//       scratchpad ring.
void scratch_ring_get_buffer(
	unsigned int     ring_number,
    ring_data_t	 	*data);

//-------------------------------------------------------------------
// scratch_ring_put_buffer
//
//    Description:
//       Put (enqueue) a buffer, length and offset tuple onto the
//       given scratchpad ring.
void scratch_ring_put_buffer(
	unsigned int    ring_number,
    ring_data_t	 data);


//-------------------------------------------------------------------
// scratch_ring_full
//
//    Description:
//       Check if the given scratchpad ring is full.
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n/a
//      Inputs: ring_number - The scratchpad ring number
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: This routine only works for scratchpad
//                  rings 0 - 11 since it uses the hardware
//                  fullness signals.  This also implies that
//                  this routine may return a full indication
//                  prior to the ring actually becoming full.
//    See also: n/a
//
//
__forceinline bool scratch_ring_full(ring_number)
{
	return inp_state_test(inp_state_scr_ring0_full + 
						  ring_number);
}

#endif // MICRO_C

#endif // SCRATCH_RINGS_H