//  Copyright (C) 2002-2003 Intel Corporation, All Rights Reserved.
//  Permission is hereby granted to merge this program code with 
//  other program material to create a derivative work.  This 
//  derivative work may be distributed in compiled object form only.
//  Any other publication of this program, in any form, without the 
//  explicit permission of the copyright holder is prohibited.
//
//  Send questions and comments to erik.j.johnson@intel.com, 
//  aaron.kunze@intel.com

//-------------------------------------------------------------------
// spi3_rx.uc - Chapter 5
// Reassembles incomming packets from a single port on the
// SPI-4 interface using a single thread
//

#ifndef spi3_RX_UC
#define spi3_RX_UC

// This file contains an implementation of the receive reassembly
// process using a single thread.  Incomming mpackets from a single
// port are assembled into complete packets.  Handles to these 
// packets are then placed on a scratch ring for processing.
//

//-------------------------------------------------------------------
// Include files
#include "rx.h"
#include "buf.uc"
#include "ixp_hw.uc"
#include "stdmac.uc"

#define RBUF_ELEM_SIZE 		128
#define RX_CONTROL_VAL		((1 << RX_EN_SPHY_BITPOS) | \
							 (1 << RBUF_ELE_SIZE_0_BITPOS))

//-------------------------------------------------------------------
// _spi3_rx_free_rbuf
//
//    Description:
//       Place the given RBUF element onto the RBUF freelist
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n.a
//      Inputs: in_elem	- The RBUF element number to free
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: Writes the given element number into
//                  the MSF RBUF done address.  The RBUF
//                  should not be used after calling this macro
//
//    See also: n/a
//
#macro _spi3_rx_free_rbuf(in_elem)
.begin
	.reg val addr
	// The element number is placed into the upper 16 
	// bits as required by the MSF fast_wr instruction
	shf_left(val, in_elem, 16)
	immed32(addr, MSF_RBUF_ELEMENT_DONE_ADDR)
	msf[fast_wr, --, addr, val]
.end
#endm

//-------------------------------------------------------------------
// spi3_rx_init
//
//    Description:
//       Initialize the appropriate CSRs to receive packets on
//       the spi3 interface.
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n/a
//      Inputs: n/a
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: Writes to the MSF receive control and 
//                  port map CSRs
//
//    See also: n/a
//
#macro spi3_rx_init()
.begin
	.if (ctx() == 0)
		.reg addr // Holds the address of the CSR being set

		.reg $rx_ctl_xfer
		.sig msf_rx_ctl_sig

		//----------- Set the receive control CSR in the MSF.
		immed32($rx_ctl_xfer, 0x00090004)
		immed32(addr, MSF_RX_CONTROL_ADDR)
		msf[write, $rx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_rx_ctl_sig]

		// Free all RBUF elements
		.begin
			.reg elem

			immed32(elem, 0)
			.while(elem < 64/4)
				_spi3_rx_free_rbuf(elem)
				add(elem, elem, 1)
			.endw
		.end


		//----------- Set the rx up CSRs in the MSF.
		immed32($rx_ctl_xfer, 0x39)
		immed32(addr, 0x80)
		msf[write, $rx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_rx_ctl_sig]
		immed32($rx_ctl_xfer, 0x39)
		immed32(addr, 0x84)
		msf[write, $rx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_rx_ctl_sig]
		immed32($rx_ctl_xfer, 0x39)
		immed32(addr, 0x88)
		msf[write, $rx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_rx_ctl_sig]
		immed32($rx_ctl_xfer, 0x39)
		immed32(addr, 0x8c)
		msf[write, $rx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_rx_ctl_sig]


		//----------- Set the receive control CSR in the MSF.
		immed32($rx_ctl_xfer, 0xf0090004)
		immed32(addr, MSF_RX_CONTROL_ADDR)
		msf[write, $rx_ctl_xfer, addr, 0, 1], 
			ctx_swap[msf_rx_ctl_sig]
	.endif
.end
#endm // rx_csr_init


//-------------------------------------------------------------------
// _spi3_rx_get_mpacket
//
//    Description:
//       Add the thread into the receive thread freelist and wait
//		 for an mpacket to arrive
//
//    Parameters:
//      Outputs: out_rsw0 - The first of two ordered SRAM transfer
//							  registers where the receive status
//							  control word will be placed 
//      		 out_rsw1 - The second of two ordered SRAM transfer
//							  registers where the receive status
//							  control word will be placed 
//      In/Outs: n/a
//      Inputs: n/a
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: n/a
//    See also: n/a
//
//
#macro _spi3_rx_get_mpacket(out_rsw0, out_rsw1)
.begin
	.sig rx_complete_sig
	.reg rx_tfl rx_tfl_addr context

	// Add the current thread into the thread freelist
	.set_sig rx_complete_sig
	// Add the wakeup signal when an mpacket arrives
	shf_left(rx_tfl, &rx_complete_sig, 12)
	// Add the microengine number to signal
	alu_shf_left(rx_tfl, rx_tfl, OR, __UENGINE_ID, 7)
	// Add the context to signal
	local_csr_rd[ACTIVE_CTX_STS]
	immed[context, 0]
	alu[context, 0x7, and, context]	
	alu_shf_left(rx_tfl, rx_tfl, OR, context, 4)
	// Add the transfer register address 
	// where the RSW words should be placed
	alu_shf_left(rx_tfl, rx_tfl, OR, &out_rsw0, 0)

	// Place the data into the upper 16 bits for
	// the fast_wr operation
	shf_left(rx_tfl, rx_tfl, 16)
	immed32(rx_tfl_addr, MSF_RX_THREAD_FREELIST_0_ADDR)
	msf[fast_wr, --, rx_tfl_addr, rx_tfl]

	.set out_rsw0 out_rsw1
	ctx_arb[rx_complete_sig] // wait for an mpacket
.end
#endm


//-------------------------------------------------------------------
// _spi3_rx_move_rbuf_to_dram
//
//    Description:
//       Transfer the given RBUF element into the given DRAM
// 		 address
//
//    Parameters:
//      Outputs: n/a 
//      In/Outs: n/a
//      Inputs: in_rbuf_elem - The RBUF element number to transfer
//              in_dram_addr - The address to transfer to RBUF
//							   into
//				in_size - The number of bytes to transfer
//				in_dram_sig	 - The signal to use for the transfer
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: The given dual signal must be caught by the
//					calling routine.
//					The in_rbuf_elem_size must be between 1 and
//					128.  The number of bytes transfered will
//					be (in_rbuf_elem_size >> 3) << 3) since
//					the transfer must be an even number of quadwords
//    See also: n/a
//
//    Example Usage:
//		.sig dram_signal
//		rx_move_rbuf_to_dram(2, 0x20, 128, dram_signal)
//
#macro _spi3_rx_move_rbuf_to_dram(in_rbuf_elem, in_dram_addr, \
							in_size, in_dram_sig)
.begin
	.reg indir rbuf_addr qwords_to_xfer new_size

	// Compute the RBUF address.  This is the base RBUF
	// address in the MSF plus the element number times
	// 128.  The multiplication by 128 comes from the fact
	// that the element number given in the RSW is 
	// divided by 64, plus the size of the RBUF element 
	immed32(rbuf_addr, MSF_RBUF_BASE_ADDR)
	alu_shf_left(rbuf_addr, rbuf_addr, +, 
				 in_rbuf_elem, 7)

	// Override the rbuf addr
	shf_left(indir, 1, 4) 
	alu_shf_left(indir, indir, OR, rbuf_addr, 5)
	// Override the transfer size
	alu_shf_left(indir, indir, OR, 1, 25)
	add(new_size, in_size, 7)
	alu_shf_right(qwords_to_xfer, 0xff, AND, new_size, 3)
	sub(qwords_to_xfer, qwords_to_xfer, 1) 
	alu_shf_left(indir, indir, OR, qwords_to_xfer, 21)
	dram[rbuf_rd, --, in_dram_addr, 0, max_16], 
		indirect_ref, 
		sig_done[in_dram_sig]
	.use indir // Suppress an assembler warning
.end
#endm

//-------------------------------------------------------------------
// spi3_rx
//
//    Description:
//       Reassemble an incomming packet. 
//
//    Parameters:
//      Outputs: (implicit)
//				 dl_buf_hndle - a handle representing the
//               		properly reassembled packet
//               dl_meta_buf_size - the length, in bytes, of the
//                  	reassembled packet
//      In/Outs: n/a
//      Inputs: n/a
//      Constants: n/a
//      Labels: n/a
//
//    Side effects:  
//		This routine requires that packets are arriving on only one 
//      port of the spi3 interface.  If this is not the case, 
//		mpackets from multiple ports will be intermixed into the 
//		one final packet.
//
//		This routine does not return until a packet is reassembled
//      without error.
//
//    See also: spi3_rx_init()
//
#macro spi3_rx()
.begin
	.reg buf_length
	.reg cur_mpacket_addr // A pointer into dram where 
						  // the next mpacket should be 
						  // placed
	.set cur_mpacket_addr
	.reg rbuf_elem, elem_size
						  // The RBUF element number 
						  // and size of the current 
						  // mpacket.  

	.reg $rsw0 $rsw1	  // The receive status words
	.xfer_order $rsw0 $rsw1

	.sig buf_alloc_sig

	immed32(dl_buf_handle, 0)
	immed32(buf_length, 0)

	.while(1)
		// Get the next mpacket
		_spi3_rx_get_mpacket($rsw0, $rsw1)
		.use $rsw1 // Suppress an assembler warning

		// Extract the RBUF element number and size
		alu_shf_right(rbuf_elem, 
					  RSW_SPHY3_ELEMENT_MASK, AND,
		  			  $rsw0, RSW_SPHY3_ELEMENT_BITPOS)
		alu_shf_right(elem_size, 
					  RSW_SPHY3_BYTECOUNT_MASK,
					  AND, $rsw0, 
					  RSW_SPHY3_BYTECOUNT_BITPOS)

		// Check for errors in the packet
		// These indicate that the current buffer, 
		// if any, should be discarded
		.if (BIT($rsw0, RSW_SPHY3_ERRORS_BITPOS))
			.if (dl_buf_handle != 0)
				// Drop the packet
				dl_buf_drop(dl_buf_handle)
			.endif

			_spi3_rx_free_rbuf(rbuf_elem)
			immed32(dl_buf_handle, 0)
			immed32(buf_length, 0)
			.continue
		.endif
	
		// If this is the SOP, allocate a new buffer		
		.if (BIT($rsw0, RSW_SPHY3_SOP_BITPOS) == 1)
			.if (dl_buf_handle == 0)
				.begin
					.reg $buf_handle_xfer
					dl_buf_alloc($buf_handle_xfer, 
							 	 BUF_QARRAY_BASE,
							 	 buf_alloc_sig,
							 	 buf_alloc_sig)
					move(dl_buf_handle,
						 $buf_handle_xfer)
				.end
				.if (dl_buf_handle == 0)
					// No more buffers
					_spi3_rx_free_rbuf(rbuf_elem)
					.continue
				.endif
			.endif
			dl_buf_get_data(cur_mpacket_addr,
							dl_buf_handle)
		.elif (dl_buf_handle == 0)
			// An MOP or EOP mpacket was received 
			// without an SOP mpacket first
			_spi3_rx_free_rbuf(rbuf_elem)
			.continue
		.endif

		// Move the mpacket into DRAM
		.begin
			.sig rbuf_to_dram_sig 

			_spi3_rx_move_rbuf_to_dram(
					rbuf_elem, 
					cur_mpacket_addr,
					elem_size, 
					rbuf_to_dram_sig)

			// Update the buffer length
			add(buf_length, buf_length, elem_size)

			// Wait for the mpacket to move into DRAM
			ctx_arb[rbuf_to_dram_sig]

			_spi3_rx_free_rbuf(rbuf_elem)

			// If this is the EOP mpacket then return
			.if (BIT($rsw0, RSW_SPHY3_EOP_BITPOS) == 1)
					.break
			.endif

			// Update the reassembly pointer
			add(cur_mpacket_addr, cur_mpacket_addr, 
				elem_size)
		.end
	.endw

	dl_meta_set_offset(0)
	dl_meta_set_buffer_size(buf_length)
	immed32(dl_next_block, SPI3_RX_NEXT_BLOCK)
.end
#endm // rx_packet

#endif // spi3_RX_UC