//  Copyright (C) 2002-2003 Intel Corporation, All Rights Reserved.
//  Permission is hereby granted to merge this program code with 
//  other program material to create a derivative work.  This 
//  derivative work may be distributed in compiled object form only.
//  Any other publication of this program, in any form, without the 
//  explicit permission of the copyright holder is prohibited.
//
//  Send questions and comments to erik.j.johnson@intel.com, 
//  aaron.kunze@intel.com

//-------------------------------------------------------------------
// scratch_rings.h - Chapter 5
// Common utilities for accessing the scratch rings used between
// rx, processing and tx
//

#ifndef SCRATCH_RINGS_UC
#define SCRATCH_RINGS_UC

#include "scratch_rings.h"
#include "xbuf.uc"

#if (RX_TO_PROCESSING_RING > 11)
#error			"For Ring Number Greater than 11, we cannot use the SCR_Ring#_Full input state. Check RX_TO_PROCESSING_RING"
#endif


//-------------------------------------------------------------------
// scratch_ring_init
//
//    Description:
//       Initialize the given scratch ring
//
//    Parameters:
//      Outputs: n/a 
//      In/Outs: n/a
//      Inputs: IN_RING_NUM - The scratchpad ring number
//				IN_RING_BASE - The base address in scratchpad memory
//							   for the ring data
//				IN_RING_SIZE - The number of long words for the ring
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: n/a
//    See also: n/a
//
#macro scratch_ring_init(IN_RING_NUM, IN_RING_BASE, \
                         IN_RING_SIZE)
.begin
	.reg $ring_init_xfer $ring_head $ring_tail ring_init
	.sig ring_init_sig ring_head_sig ring_tail_sig

	immed32(ring_init, IN_RING_BASE)
	alu_shf_left($ring_init_xfer, ring_init, OR,
				 ((IN_RING_SIZE/128)-1),
				 RING_BASE_SIZE_BITPOS)

	cap[write, $ring_init_xfer, 
		SCRATCH_RING_BASE_/**/IN_RING_NUM], 
		sig_done[ring_init_sig]

	immed32($ring_head, 0)
	cap[write, $ring_head, 
		SCRATCH_RING_HEAD_/**/IN_RING_NUM], 
		sig_done[ring_head_sig]

	immed32($ring_tail, 0)
	cap[write, $ring_tail, 
		SCRATCH_RING_TAIL_/**/IN_RING_NUM], 
		sig_done[ring_tail_sig]

	ctx_arb[ring_init_sig, ring_head_sig, ring_tail_sig]
.end
#endm


//-------------------------------------------------------------------
// scratch_ring_get_buffer
//
//    Description:
//       Dequeue a buffer, length and offset tuple from the given
//       scratchpad ring.
//
//    Parameters:
//      Outputs: handle - the buffer handle
//               length - the length, in bytes, of the buffer
//               offset - the offset of the start of the actual data
//                        within the buffer, in bytes 
//      In/Outs: n/a
//      Inputs: IN_RING_NUM - The scratchpad ring number
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: n/a
//    See also: n/a
//
//    Example Usage:
//		ring_get_buffer(3, buf_handle, buf_len, buf_offset)
//
#macro scratch_ring_get_buffer(\
					   IN_RING_NUM, \
                       out_handle, \
				       out_length, \
					   out_offset)
.begin
	.reg ring_addr
	.sig ring_signal

	xbuf_alloc($ring_data, 3, read)

	immed32(ring_addr, (IN_RING_NUM<<2))
	scratch[get, $ring_data[0], 
			ring_addr, 0, 3],
		ctx_swap[ring_signal]
	move(out_handle, $ring_data[0])
	move(out_length, $ring_data[1])
	move(out_offset, $ring_data[2])

	xbuf_free($ring_data)
.end
#endm



//-------------------------------------------------------------------
// scratch_ring_put_buffer
//
//    Description:
//       Put (enqueue) a buffer, length and offset tuple onto the
//       given scratchpad ring.
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n/a
//      Inputs: IN_RING_NUM - The scratchpad ring number
//              handle - the buffer handle
//              length - the length, in bytes, of the buffer
//              offset - the offset of the start of the actual data
//                       within the buffer, in bytes 
//      Constants: n/a
//      Labels: n/a
//
//    Side effects: n/a
//    See also: n/a
//
//    Example Usage:
//		ring_put_buffer(3, buf_handle, buf_len, buf_offset)
//
#macro scratch_ring_put_buffer(\
					   IN_RING_NUM, \
                       in_handle, \
				       in_length, \
					   in_offset)
.begin
	.reg ring_addr
	.sig ring_signal

	xbuf_alloc($ring_data, 3, write)

	immed32(ring_addr, (IN_RING_NUM<<2))

	move($ring_data[0], in_handle)
	move($ring_data[1], in_length)
	move($ring_data[2], in_offset)
	scratch[put, $ring_data[0], 
			ring_addr, 0, 3],
		ctx_swap[ring_signal]

	xbuf_free($ring_data)
.end
#endm

//-------------------------------------------------------------------
// scratch_ring_full
//
//    Description:
//       Check if the given scratchpad ring is full.
//
//    Parameters:
//      Outputs: n/a
//      In/Outs: n/a
//      Inputs: n/a
//      Constants: IN_RING_NUM - The scratchpad ring number
//      Labels: IN_FULL_TARGET - The target to branch to if the
//							 ring is full
//
//    Side effects: This routine only works for scratchpad
//                  rings 0 - 11 since it uses the hardware
//                  fullness signals.  This also implies that
//                  this routine may return a full indication
//                  prior to the ring actually becoming full.
//    See also: n/a
//
//    Example Usage:
//		ring_full(3, full_target#)
//
#macro scratch_ring_full(IN_RING_NUM, \
						 IN_FULL_TARGET)
	br_inp_state[SCR_Ring/**/IN_RING_NUM/**/_Full, 
				 IN_FULL_TARGET]
#endm


#endif // RINGS_UC